/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: CifXDeviceChannel.h 12750 2019-03-01 10:16:35Z LuisContreras $:

  Description:
    cifX Channel handler class. Connects and services a channel on a cifX Device

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2010-04-29  Updated to latest cifXUser / stdint.h
    2006-06-28  initial version

**************************************************************************************/

#pragma once

///////////////////////////////////////////////////////////////////////////////////////////
/// \file CifXDeviceChannel.h
///   cifX Channel handler class. Connects and services a channel on a cifX Device
///////////////////////////////////////////////////////////////////////////////////////////

#include "cifxdevicebase.h"

///////////////////////////////////////////////////////////////////////////////////////////
/// cifX Channel handler class
///////////////////////////////////////////////////////////////////////////////////////////
class CCifXDeviceChannel : public CCifXDeviceBase
{
public:
  CCifXDeviceChannel(uint32_t ulChannel, BOARD_INFORMATION* ptBoardInfo, CHANNEL_INFORMATION* ptChannelInfo);
  virtual ~CCifXDeviceChannel(void);

  virtual bool     IsSystemDevice(void);
  virtual int32_t  OpenDevice(void);
  virtual int32_t  CloseDevice(void);
  virtual int32_t  PutPacket(CIFX_PACKET* ptPacket, uint32_t ulTimeout);
  virtual int32_t  GetPacket(CIFX_PACKET* ptPacket, uint32_t ulBufferSize, uint32_t ulTimeou);
  CCifXDeviceBase::PROPERTY_VECTOR GetChannelInformation(uint32_t ulCmd);
  virtual int32_t  GetMBXState(uint32_t* pulRecvCount, uint32_t* pulSendCount);
  virtual int32_t  Reset(uint32_t ulMode, uint32_t ulTimeout, uint32_t ulResetParam);
  virtual int32_t  Upload(char* szFilename, uint32_t ulChannel, unsigned char* pbBuffer, uint32_t* pulFileSize);


  virtual int32_t          IOInfo(uint32_t ulAreaNumber, 
                               uint32_t ulOffset, 
                               uint32_t ulDataLen, 
                               void* pvData);

  virtual int32_t          IORead(uint32_t ulAreaNumber, 
                               uint32_t ulOffset, 
                               uint32_t ulDataLen, 
                               void* pvData, 
                               uint32_t ulTimeout);

  virtual int32_t          IOWrite(uint32_t ulAreaNumber, 
                               uint32_t ulOffset, 
                               uint32_t ulDataLen, 
                               void* pvData, 
                               uint32_t ulTimeout);
  virtual int32_t          IOReadSend(uint32_t ulAreaNumber, 
                                   uint32_t ulOffset, 
                                   uint32_t ulDataLen, 
                                   void* pvData);

  virtual int32_t          Watchdog(uint32_t ulCmd, uint32_t* pulTrigger);
  virtual int32_t          HostState(uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout);
  virtual int32_t          BusState(uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout);
  virtual int32_t          DMAState(uint32_t ulCmd, uint32_t* pulState);
  virtual int32_t          Download(uint32_t ulChannel, uint32_t ulMode, char* szFileName, unsigned char* pabFileData, uint32_t ulFileSize, PFN_PROGRESS_CALLBACK pfnCallback, void* pvUser);
  virtual int32_t          ConfigLock(uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout);

  virtual int32_t          FindFirstFile(uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo);
  virtual int32_t          FindNextFile(uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo);


  std::vector<CCifXDeviceBase::PROPERTY_LIST> GetDescription(void);
  CString                                     GetChannelInfoName(void);

  virtual uint32_t    GetChannelNr(void)  { return m_ulChannel; }

  /////////////////////////////////////////////////////////////////////////////
  /// Return the name of the channel (e.g. "CifX0 (alias) Channel0")
  ///   \return name of the cannel (used for Caption of main window)
  /////////////////////////////////////////////////////////////////////////////
  virtual CString GetDeviceName(void)
  {
    CString csRet(CCifXDeviceBase::GetDeviceName());
    csRet.AppendFormat(_T(" Channel%u"), m_ulChannel);
    
    return csRet;    
  }

  /////////////////////////////////////////////////////////////////////////////
  /// Return the short name of the channel (e.g. "Channel0")
  ///   \return name of the cannel (used for Tree in open dialog)
  /////////////////////////////////////////////////////////////////////////////
  virtual CString GetShortDeviceName(void)
  {
    CString csRet;
    csRet.Format(_T(" Channel%u"), m_ulChannel);
    
    return csRet;    
  }

protected:
  HANDLE   m_hChannel;   //!< cifX driver handle to the channel
  uint32_t m_ulChannel;  //!< number of the channel, from cifX device driver enumeration
};
